/*
 * Geekdroid is a utility library for development on the Android
 * Platform.
 *
 * Copyright (C) 2017-2023 by Frederic-Charles Barthelery.
 *
 * This file is part of Geekdroid.
 *
 * Geekdroid is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Geekdroid is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Geekdroid.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.geekorum.geekdroid.osslicenses

import android.annotation.SuppressLint
import android.content.Context
import okio.ByteString
import okio.buffer
import okio.source
import java.io.InputStream

/**
 * Parse licences data generated by the "com.google.android.gms.oss-licenses-plugin" gradle plugin.
 * [thirdPartyLicensesInput] is usually res/raw/third_party_licenses file
 * [thirdPartyLicensesMetadataInput] is usually res/raw/third_party_license_metadata file
 */
class OssLicenseParser(
    private val thirdPartyLicensesInput: InputStream,
    private val thirdPartyLicensesMetadataInput: InputStream
) {
    fun parseLicenses(): Map<String, String> {
        val licenses = readLicensesFile()
        return buildLicenseInfo(licenses)
    }

    private fun readLicensesFile(): ByteString {
        return thirdPartyLicensesInput.source().use { source ->
            source.buffer().use {
                it.readByteString()
            }
        }
    }

    private fun buildLicenseInfo(license: ByteString): Map<String, String> {
        return thirdPartyLicensesMetadataInput.source().use { source ->
            source.buffer().use {
                buildMap {
                    while (true) {
                        val line = it.readUtf8Line() ?: break
                        if (line.isNotBlank()) {
                            with(line.toLineParser()) {
                                val start = readStartIdx()
                                val length = readLength()
                                val dependency = readName()
                                val licenseTxt = license.substring(
                                    beginIndex = start,
                                    endIndex = start + length + 1
                                ).string(Charsets.UTF_8)
                                put(dependency, licenseTxt)
                            }
                        }
                    }
                }
            }
        }
    }

    companion object {
        @SuppressLint("DiscouragedApi")
        fun openDefaultThirdPartyLicenses(context: Context): InputStream {
            val thirdPartyLicensesId = context.resources.getIdentifier("third_party_licenses", "raw", context.packageName)
            check(thirdPartyLicensesId != 0) { "third_party_licenses was not found in resources raw of ${context.packageName}"}
            return context.resources.openRawResource(thirdPartyLicensesId)
        }

        @SuppressLint("DiscouragedApi")
        fun openDefaultThirdPartyLicensesMetadata(context: Context): InputStream {
            val thirdPartyLicensesMetadataId = context.resources.getIdentifier("third_party_license_metadata", "raw", context.packageName)
            check(thirdPartyLicensesMetadataId != 0) { "third_party_license_metadata was not found in resources raw of ${context.packageName}"}
            return context.resources.openRawResource(thirdPartyLicensesMetadataId)
        }
    }
}

private class LicenseMetadataLineParser(
    private val line: String
) {

    private var idx = 0

    fun readStartIdx(): Int {
        val end = line.indexOf(':', startIndex = idx)
        val result = line.substring(idx, end).toInt()
        idx = end + 1
        return result
    }

    fun readLength(): Int {
        val end = line.indexOf(' ', startIndex = idx)
        val result = line.substring(idx, end).toInt()
        idx = end + 1
        return result
    }

    fun readName(): String {
        val result = line.substring(idx)
        idx = line.length + 1
        return result
    }

}

private fun String.toLineParser() = LicenseMetadataLineParser(this)
