/**
 *
 */
package com.beem.project.beem.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jivesoftware.smack.Roster;
import org.jivesoftware.smack.RosterEntry;
import org.jivesoftware.smack.RosterGroup;
import org.jivesoftware.smack.RosterListener;
import org.jivesoftware.smack.XMPPException;
import org.jivesoftware.smack.packet.Presence;
import org.jivesoftware.smack.util.StringUtils;

import android.os.RemoteCallbackList;
import android.os.RemoteException;
import android.util.Log;

import com.beem.project.beem.service.aidl.IBeemRosterListener;

/**
 * This class implement a Roster adapter for BEEM.
 * @author darisk
 */
public class RosterAdapter extends com.beem.project.beem.service.aidl.IRoster.Stub {

    private static final String TAG = "RosterAdapter";
    private Roster mAdaptee;
    private RemoteCallbackList<IBeemRosterListener> mRemoteRosListeners = new RemoteCallbackList<IBeemRosterListener>();

    private Map<String, Contact> mContacts = new HashMap<String, Contact>();

    private RosterListenerAdapter mRosterListener = new RosterListenerAdapter();

    /**
     * Constructor.
     * @param roster the roster to adapt
     */
    public RosterAdapter(final Roster roster) {
	mAdaptee = roster;
	roster.addRosterListener(mRosterListener);
	for (RosterEntry entry : roster.getEntries()) {
	    String user = StringUtils.parseBareAddress(entry.getUser());
	    if (!mContacts.containsKey(user)) {
		Contact c = getContactFromRosterEntry(entry);
		mContacts.put(user, c);
	    }
	}
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void addRosterListener(IBeemRosterListener listen) throws RemoteException {
	if (listen != null)
	    mRemoteRosListeners.register(listen);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Contact addContact(String user, String name, String[] groups) throws RemoteException {
	Contact res = null;
	RosterEntry contact = mAdaptee.getEntry(user);
	if (contact != null) {
	    res = mContacts.get(user);
	} else {
	    try {
		mAdaptee.createEntry(user, name, groups);
		contact = mAdaptee.getEntry(user);
		res = new Contact(user);
		mContacts.put(user, res);
	    } catch (XMPPException e) {
		Log.e(TAG, "Error while adding new contact", e);
		return null;
	    }
	}
	if (groups != null) {
	    for (String groupStr : groups) {
		RosterGroup group = mAdaptee.getGroup(groupStr);
		if (group == null) {
		    group = mAdaptee.createGroup(groupStr);
		}		    
		try {
		    group.addEntry(contact);
		} catch (XMPPException e) {
		    e.printStackTrace();
		}
		res.addGroup(groupStr);
	    }		
	}
	return res;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void deleteContact(Contact contact, String group) throws RemoteException {
	try {
	    RosterEntry entry = mAdaptee.getEntry(contact.getJID());
	    if (entry.getGroups().size() == 0) {
		mContacts.remove(contact.getJID());
		mAdaptee.removeEntry(entry);
	    } else {
		mContacts.get(contact.getJID()).delGroup(group);
		mAdaptee.getGroup(group).removeEntry(entry);
		mRosterListener.onEntryDeleteFromGroup(group, contact.getJID());
	    }
	} catch (XMPPException e) {
	    e.printStackTrace();
	}
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void createGroup(String groupname) throws RemoteException {
	if (!mAdaptee.contains(groupname))
	    mAdaptee.createGroup(groupname);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Contact getContact(String jid) throws RemoteException {
	return mContacts.get(jid);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Contact> getContactList() throws RemoteException {
	List<Contact> res = new ArrayList<Contact>();
	res.addAll(mContacts.values());
	return res;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<String> getGroupsNames() throws RemoteException {
	Collection<RosterGroup> groups = mAdaptee.getGroups();
	ArrayList<String> result = new ArrayList<String>(groups.size());
	for (RosterGroup rosterGroup : groups) {
	    result.add(rosterGroup.getName());
	}
	return result;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void removeRosterListener(IBeemRosterListener listen) throws RemoteException {
	if (listen != null)
	    mRemoteRosListeners.unregister(listen);
    }

    /**
     * 
     */
    @Override
    public void setContactName(String jid, String name) throws RemoteException {
	mContacts.get(jid).setName(name);
	mAdaptee.getEntry(jid).setName(name);
    }

    @Override
    public PresenceAdapter getPresence(String jid) throws RemoteException {
	return new PresenceAdapter(mAdaptee.getPresence(jid));
    }

    private Contact getContactFromRosterEntry(RosterEntry entry) {
	String user = StringUtils.parseBareAddress(entry.getUser());
	Contact c = new Contact(user);
	c.setStatus(mAdaptee.getPresence(user));
	c.setGroups(entry.getGroups());
	c.setName(entry.getName());
	return c;
    }

    /**
     * Listener for the roster events. It will call the remote listeners registered.
     * @author darisk
     */
    private class RosterListenerAdapter implements RosterListener {

	/**
	 * Constructor.
	 */
	public RosterListenerAdapter() {
	}

	public void onEntryDeleteFromGroup(String group, String jid) {
	    Log.i(TAG, "entry delete listener");
	    final int n = mRemoteRosListeners.beginBroadcast();
	    for (int i = 0; i < n; i++) {
		IBeemRosterListener listener = mRemoteRosListeners.getBroadcastItem(i);
		try {
		    listener.onEntryDeleteFromGroup(group, jid);
		} catch (RemoteException e) {
		    Log.w(TAG, "entry delete listener", e);
		}
	    }
	    mRemoteRosListeners.finishBroadcast();

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void entriesAdded(Collection<String> addresses) {
	    Log.i(TAG, "Ajout de l'entry " + addresses.size() + " " + addresses.toArray()[0]);
	    final int n = mRemoteRosListeners.beginBroadcast();

	    List<String> tab = new ArrayList<String>();
	    tab.addAll(addresses);
	    for (int i = 0; i < n; i++) {
		IBeemRosterListener listener = mRemoteRosListeners.getBroadcastItem(i);
		try {
		    listener.onEntriesAdded(tab);
		} catch (RemoteException e) {
		    Log.w(TAG, "Error while adding roster entries", e);
		}
	    }
	    mRemoteRosListeners.finishBroadcast();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void entriesDeleted(Collection<String> addresses) {
	    Log.i(TAG, "Suppression de l'entry");
	    final int n = mRemoteRosListeners.beginBroadcast();

	    List<String> tab = new ArrayList<String>();
	    tab.addAll(addresses);
	    for (int i = 0; i < n; i++) {
		IBeemRosterListener listener = mRemoteRosListeners.getBroadcastItem(i);
		try {
		    listener.onEntriesDeleted(tab);
		} catch (RemoteException e) {
		    Log.w(TAG, "Error while deleting roster entries", e);
		}
	    }
	    mRemoteRosListeners.finishBroadcast();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void entriesUpdated(Collection<String> addresses) {
	    Log.i(TAG, "Update de l'entry " + addresses.size() + " " + addresses.toArray()[0]);
	    final int n = mRemoteRosListeners.beginBroadcast();

	    List<String> tab = new ArrayList<String>();
	    tab.addAll(addresses);
	    for (int i = 0; i < n; i++) {
		IBeemRosterListener listener = mRemoteRosListeners.getBroadcastItem(i);
		try {
		    listener.onEntriesUpdated(tab);
		} catch (RemoteException e) {
		    Log.w(TAG, "Error while updating roster entries", e);
		}
	    }
	    mRemoteRosListeners.finishBroadcast();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void presenceChanged(Presence presence) {
	    Log.i(TAG, "presence Changed");
	    String user = StringUtils.parseBareAddress(presence.getFrom());
	    Contact c = mContacts.get(StringUtils.parseBareAddress(user));
	    if (c == null) {
		c = new Contact(user);
		mContacts.put(user, c);
	    }
	    c.addRes(StringUtils.parseResource(presence.getFrom()));
	    c.setStatus(mAdaptee.getPresence(presence.getFrom()));
	    c.setMsgState(presence.getStatus());
	    c.setName(mAdaptee.getEntry(user).getName());
	    /* redispatch vers les IBeemRosterListener */
	    final int n = mRemoteRosListeners.beginBroadcast();

	    for (int i = 0; i < n; i++) {
		IBeemRosterListener listener = mRemoteRosListeners.getBroadcastItem(i);
		try {
		    listener.onPresenceChanged(new PresenceAdapter(presence));
		} catch (RemoteException e) {
		    // The RemoteCallbackList will take care of removing the
		    // dead listeners.
		    Log.w(TAG, "Error while updating roster entries", e);
		}
	    }
	    mRemoteRosListeners.finishBroadcast();
	}
    }

}
