/*
    BEEM is a videoconference application on the Android Platform.

    Copyright (C) 2009 by Frederic-Charles Barthelery,
                          Jean-Manuel Da Silva,
                          Nikita Kozlov,
                          Philippe Lago,
                          Jean Baptiste Vergely,
                          Vincent Veronis.

    This file is part of BEEM.

    BEEM is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    BEEM is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with BEEM.  If not, see <http://www.gnu.org/licenses/>.

    Please send bug reports with examples or suggestions to
    contact@beem-project.com or http://dev.beem-project.com/
 */
package com.beem.project.beem.ui;

import java.io.IOException;

import android.app.Activity;
import android.content.ContentUris;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.database.ContentObserver;
import android.database.Cursor;
import android.graphics.BitmapFactory;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.LayerDrawable;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.provider.ContactsContract;
import android.provider.ContactsContract.RawContacts;
import android.util.Log;
import android.view.ContextMenu;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewStub;
import android.widget.AdapterView;
import android.widget.AdapterView.AdapterContextMenuInfo;
import android.widget.AdapterView.OnItemClickListener;
import android.widget.Filterable;
import android.widget.Gallery;
import android.widget.ImageView;
import android.widget.ListView;
import android.widget.SimpleCursorAdapter;
import android.widget.TextView;

import com.beem.project.beem.BeemApplication;
import com.beem.project.beem.BeemIntent;
import com.beem.project.beem.R;
import com.beem.project.beem.providers.MUCs;
import com.beem.project.beem.service.BeemAvatarCache;
import com.beem.project.beem.ui.wizard.AccountConfigure;

/**
 * The contact list activity displays the roster of the user.
 * @author Vincent V. <marseille@beem-project.com>
 */
public class ContactList extends Activity {

    private static final String SETTINGS_HIDDEN_CONTACT = "settings_key_hidden_contact";
    private static final String TAG = "ContactList";

    private final int BANNER_POS_ALL_CONTACT = 0;
    private int BANNER_POS_NO_GROUP;
    private int BANNER_POS_MUC;

    private final BeemContactListOnClick mOnContactClick = new BeemContactListOnClick();
    private final BeemMucListOnClick mOnMucClick = new BeemMucListOnClick();

    private SharedPreferences mSettings;
    private BeemBanner mAdapterBanner;
    private SimpleCursorAdapter mAdapterContactList;

    private String mAccountName;

    /**
     * Constructor.
     */
    public ContactList() {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void onCreate(Bundle saveBundle) {
	super.onCreate(saveBundle);

	setContentView(R.layout.contactlist);

	Bundle b = getIntent().getExtras();
	if (b == null) {
	    //TODO: Add toast to advice need to configure at least 1 beem account (Should not happend)
	    startActivity(new Intent(ContactList.this, AccountConfigure.class));
	}
	mAccountName = b.getString(BeemIntent.EXTRA_ACCOUNT);
	if (mAccountName == null) {
	    //TODO: Add toast to advice need to configure at least 1 beem account (Should not happend)
	    startActivity(new Intent(ContactList.this, AccountConfigure.class));
	}
	mSettings = getSharedPreferences(mAccountName, MODE_PRIVATE);

	// Get Groups list
	final Cursor cursorGroup = getContentResolver().query(ContactsContract.Groups.CONTENT_URI,
	    new String[] { ContactsContract.Groups._ID, ContactsContract.Groups.TITLE },
	    ContactsContract.Groups.ACCOUNT_NAME + "=?", new String[] { mAccountName }, null);

	mAdapterBanner = new BeemBanner(this, R.layout.contactlist_group, cursorGroup, new String[] {
	    ContactsContract.Groups._ID, ContactsContract.Groups.TITLE }, new int[] { R.id.contactlist_group });
	cursorGroup.registerContentObserver(new BeemGroupObserver(new Handler()));

	BANNER_POS_MUC = cursorGroup.getCount() - 1;
	BANNER_POS_NO_GROUP = cursorGroup.getCount() - 2;

	// Get Contacts list
	Cursor cursorRawContact = getContacts();
	mAdapterContactList = new BeemContactList(this, R.layout.contactlistcontact, cursorRawContact, new String[] {
	    ContactsContract.RawContacts._ID, ContactsContract.RawContacts.CONTACT_ID,
	    ContactsContract.RawContacts.SOURCE_ID }, new int[] { R.id.avatar, R.id.contactlistpseudo,
	    R.id.contactlistmsgperso });

	ListView listView = (ListView) findViewById(R.id.contactlist);
	listView.setOnItemClickListener(mOnContactClick);
	registerForContextMenu(listView);
	listView.setAdapter(mAdapterContactList);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void onResume() {
	super.onResume();
	if (!mSettings.getBoolean("settings_key_hide_groups", false))
	    showGroups();
	else
	    hideGroups();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean onPrepareOptionsMenu(Menu menu) {
	menu.clear();
	MenuInflater inflater = getMenuInflater();
	if (mAdapterContactList.getClass() == BeemMucList.class)
	    inflater.inflate(R.menu.muc_list, menu);
	else
	    inflater.inflate(R.menu.contact_list, menu);
	return super.onPrepareOptionsMenu(menu);

    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final boolean onOptionsItemSelected(MenuItem item) {
	switch (item.getItemId()) {
	    case R.id.contact_list_menu_settings:
		Intent i = new Intent(this, Settings.class);
		i.putExtra("account_name", mAccountName);
		startActivity(i);
		return true;
	    case R.id.contact_list_menu_add_contact:
		Intent addContactIntent = new Intent(ContactList.this, AddContact.class);
		addContactIntent.putExtra("account_name", mAccountName);
		startActivity(addContactIntent);
		return true;
	    case R.id.contact_list_menu_muc_add_contact:
		Intent addMUCIntent = new Intent(ContactList.this, AddMuc.class);
		addMUCIntent.putExtra("account_name", mAccountName);
		startActivity(addMUCIntent);
		return true;
	    case R.id.menu_change_status:
		startActivity(new Intent(ContactList.this, ChangeStatus.class));
		return true;
	    case R.id.contact_list_menu_chatlist:
		//		List<Contact> openedChats;
		//		try {
		//		    openedChats = mChatManager.getOpenedChatList();
		//		    Log.d(TAG, "opened chats = " + openedChats);
		//		    Dialog chatList = new ChatList(ContactList.this, openedChats).create();
		//		    chatList.show();
		//		} catch (RemoteException e) {
		//		    e.printStackTrace();
		//		}
		return true;
	    case R.id.contact_list_menu_sync:
		Intent intent = new Intent(BeemIntent.ACTION_SYNC);
		intent.putExtra(BeemIntent.EXTRA_ACCOUNT, mAccountName);
		startService(intent);
		return true;
	    case R.id.menu_disconnect:
		return true;
	    default:
		return false;
	}
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onCreateContextMenu(ContextMenu menu, View v, ContextMenu.ContextMenuInfo menuInfo) {
	super.onCreateContextMenu(menu, v, menuInfo);
	MenuInflater inflater = getMenuInflater();
	inflater.inflate(R.menu.contactlist_context, menu);
	AdapterContextMenuInfo info = (AdapterView.AdapterContextMenuInfo) menuInfo;
	// Contact c = new Contact();
	// try {
	// mSelectedContact = mRoster.getContact(c.getJID());
	// } catch (RemoteException e) {
	// e.printStackTrace();
	// }
	// menu.setHeaderTitle(mSelectedContact.getJID());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean onContextItemSelected(MenuItem item) {
	Intent in;
	boolean result;
	switch (item.getItemId()) {
	    case R.id.contact_list_context_menu_chat_item:
		//		    List<String> res = mSelectedContact.getMRes();
		//		    if (res.isEmpty()) {
		//			result = false;
		//			break;
		//		    }
		//		    for (String resv : res) {
		//			in = new Intent(this, Chat.class);
		//			in.setData(mSelectedContact.toUri(resv));
		//			item.getSubMenu().add(resv).setIntent(in);
		//		    }
		result = true;
		break;
	    case R.id.contact_list_context_menu_call_item:
		//		    try {
		//			mXmppFacade.call(mSelectedContact.getJID() + "/psi");
		//			result = true;
		//		    } catch (RemoteException e) {
		//			e.printStackTrace();
		//		    }
		result = true;
		break;
	    case R.id.contact_list_context_menu_user_info:
		//		    item.getSubMenu().setHeaderTitle(mSelectedContact.getJID());
		result = true;
		break;
	    case R.id.contact_list_context_menu_userinfo_alias:
		//		    Dialog alias = new Alias(ContactList.this, mRoster, mSelectedContact).create();
		//		    alias.show();
		result = true;
		break;
	    case R.id.contact_list_context_menu_userinfo_group:
		//		    in = new Intent(this, GroupList.class);
		//		    in.putExtra("contact", mSelectedContact);
		//		    startActivity(in);
		result = true;
		break;
	    case R.id.contact_list_context_menu_userinfo_subscription:
		//		    Dialog subscription = new ResendSubscription(ContactList.this, mXmppFacade, mSelectedContact)
		//			.create();
		//		    subscription.show();
		result = true;
		break;
	    case R.id.contact_list_context_menu_userinfo_block:
		result = true;
		break;
	    case R.id.contact_list_context_menu_userinfo_delete:
		//		    Dialog delete = new DeleteContact(ContactList.this, mRoster, mSelectedContact).create();
		//		    delete.show();
		result = true;
		break;
	    default:
		result = super.onContextItemSelected(item);
		break;
	}
	return result;
    }

    /**
     * Show the groups view.
     */
    private void showGroups() {
	ViewStub stub = (ViewStub) findViewById(R.id.contactlist_stub);
	if (stub != null) {
	    View v = stub.inflate();
	    Gallery g = (Gallery) v.findViewById(R.id.contactlist_banner);
	    g.setOnItemClickListener(new OnItemClickGroupName());
	    g.setAdapter(mAdapterBanner);
	    if (mAdapterBanner.getCount() == 0)
		v.setVisibility(View.GONE);
	    else
		v.setVisibility(View.VISIBLE);
	}
    }

    /**
     * Hide the groups view.
     */
    private void hideGroups() {
	View v = findViewById(R.id.contactlist_groupstub);
	if (v != null)
	    v.setVisibility(View.GONE);
    }

    /**
     * Adapter contact list.
     */
    private static class BeemContactList extends SimpleCursorAdapter {

	private Context mContext;
	private int mLayout;
	private BeemAvatarCache mBeemAvatarCache;
	private Cursor mCursor;

	/**
	 * Constructor.
	 * @param context
	 * @param layout
	 * @param c
	 * @param from
	 * @param to
	 */
	public BeemContactList(Context context, int layout, Cursor c, String[] from, int[] to) {
	    super(context, layout, c, from, to);
	    mContext = context;
	    mLayout = layout;
	    mBeemAvatarCache = new BeemAvatarCache(context);
	    mCursor = c;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getCount() {
	    return mCursor.getCount();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public View newView(Context context, Cursor cursor, ViewGroup parent) {
	    final LayoutInflater inflater = LayoutInflater.from(context);
	    return inflater.inflate(mLayout, parent, false);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void changeCursor(Cursor c) {
	    super.changeCursor(c);
	    mCursor = c;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void bindView(View view, Context context, Cursor cursor) {

	    long id = cursor.getLong(cursor.getColumnIndex(ContactsContract.RawContacts.CONTACT_ID));

	    Uri rawContactUri = ContentUris.withAppendedId(RawContacts.CONTENT_URI, id);
	    Uri ContactLookupUri = ContactsContract.RawContacts.getContactLookupUri(context.getContentResolver(),
		rawContactUri);
	    if (ContactLookupUri != null) {
		Cursor cursorContact = context.getContentResolver().query(
		    ContactLookupUri,
		    new String[] { ContactsContract.Contacts._ID, ContactsContract.Contacts.DISPLAY_NAME,
			ContactsContract.Contacts.CONTACT_STATUS }, null, null, null);

		if (cursorContact.moveToFirst()) {
		    String name = cursorContact.getString(cursorContact
			.getColumnIndex(ContactsContract.Contacts.DISPLAY_NAME));
		    String msg = cursorContact.getString(cursorContact
			.getColumnIndex(ContactsContract.Contacts.CONTACT_STATUS));
		    TextView nameText = (TextView) view.findViewById(R.id.contactlistpseudo);
		    nameText.setText(name);
		    TextView msgperso = (TextView) view.findViewById(R.id.contactlistmsgperso);
		    msgperso.setText(msg);

		    String jid = cursor.getString(cursor.getColumnIndex(ContactsContract.RawContacts.SOURCE_ID));
		    view.setTag(jid);

		    ImageView img = (ImageView) view.findViewById(R.id.avatar);
		    int contactStatus = 1;
		    //TODO: set the jid instead of null then debug
		    Drawable avatar = getAvatarStatusDrawable(null);
		    img.setImageDrawable(avatar);
		    img.setImageLevel(contactStatus);

		}
		cursorContact.close();
	    }
	}

	/**
	 * Get a LayerDrawable containing the avatar and the status icon. The status icon will change with the level of
	 * the drawable.
	 * @param avatarId the avatar id to retrieve or null to get default
	 * @return a LayerDrawable
	 */
	private Drawable getAvatarStatusDrawable(String jid) {
	    Drawable avatarDrawable = null;
	    if (jid != null) {
		try {
		    byte[] bb = mBeemAvatarCache.get(jid);
		    avatarDrawable = new BitmapDrawable(BitmapFactory.decodeByteArray(bb, 0, bb.length));
		} catch (IOException e) {
		    Log.w(TAG, "Error while setting the avatar", e);
		}
	    }
	    if (avatarDrawable == null)
		avatarDrawable = mContext.getResources().getDrawable(R.drawable.beem_launcher_icon_silver);
	    LayerDrawable ld = (LayerDrawable) mContext.getResources().getDrawable(R.drawable.avatar_status);
	    ld.setLayerInset(1, 36, 36, 0, 0);
	    ld.setDrawableByLayerId(R.id.avatar, avatarDrawable);
	    return ld;
	}
    }

    /**
     * Adapter muc list.
     */
    private static class BeemMucList extends SimpleCursorAdapter {

	private Context mContext;
	private int mLayout;
	private Cursor mCursor;

	public BeemMucList(Context context, int layout, Cursor c, String[] from, int[] to) {
	    super(context, layout, c, from, to);
	    mContext = context;
	    mLayout = layout;
	    mCursor = c;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getCount() {
	    return mCursor.getCount();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public View newView(Context context, Cursor cursor, ViewGroup parent) {
	    final LayoutInflater inflater = LayoutInflater.from(context);
	    return inflater.inflate(mLayout, parent, false);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void bindView(View view, Context context, Cursor cursor) {
	    String name = cursor.getString(cursor.getColumnIndex(MUCs.NAME));
	    String jid = cursor.getString(cursor.getColumnIndex(MUCs.JID));
	    TextView nameText = (TextView) view.findViewById(R.id.contactlistpseudo);
	    nameText.setText(name);
	    TextView msgperso = (TextView) view.findViewById(R.id.contactlistmsgperso);
	    msgperso.setText(jid);
	    view.setTag(jid);
	}

    }

    /**
     * Adapter banner group list.
     */
    private static class BeemBanner extends SimpleCursorAdapter {

	private Context mContext;
	private int mLayout;

	/**
	 * Constructor.
	 * @param context
	 * @param layout
	 * @param c
	 * @param from
	 * @param to
	 */
	public BeemBanner(Context context, int layout, Cursor cursor, String[] from, int[] to) {
	    super(context, layout, cursor, from, to);
	    mContext = context;
	    mLayout = layout;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public View newView(Context context, Cursor cursor, ViewGroup parent) {
	    final LayoutInflater inflater = LayoutInflater.from(mContext);
	    return inflater.inflate(mLayout, parent, false);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void bindView(View view, Context context, Cursor cursor) {
	    String name = cursor.getString(cursor.getColumnIndex(ContactsContract.Groups.TITLE));
	    TextView nameText = (TextView) view.findViewById(R.id.contactlist_group);
	    if (nameText != null) {
		nameText.setText(name);
	    }
	    String groupid = cursor.getString(cursor.getColumnIndex(ContactsContract.Groups._ID));
	    view.setTag(groupid);
	}
    }

    /**
     * Event simple click on item of the contact list.
     */
    private class BeemContactListOnClick implements OnItemClickListener {
	/**
	 * Constructor.
	 */
	public BeemContactListOnClick() {
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onItemClick(AdapterView<?> arg0, View v, int pos, long lpos) {
	    String jid = (String) v.getTag();
	    Log.i(TAG, "OnClick Jid :" + jid);
	    Intent i = new Intent(ContactList.this, Chat.class);
	    i.setData(Uri.parse("imto://jabber/" + jid));
	    startActivity(i);
	}
    }

    private class BeemMucListOnClick implements OnItemClickListener {
	/**
	 * Constructor.
	 */
	public BeemMucListOnClick() {
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onItemClick(AdapterView<?> arg0, View v, int pos, long lpos) {
	    String jid = (String) v.getTag();
	    Log.i(TAG, "OnClick MUC Jid:" + jid);
	    Intent intent = new Intent(BeemIntent.ACTION_MUC_JOIN);
	    intent.putExtra(BeemIntent.EXTRA_ACCOUNT, mAccountName);
	    intent.putExtra(BeemIntent.EXTRA_JID, jid);
	    startService(intent);

	    Intent i = new Intent(ContactList.this, MucChat.class);
	    i.setData(Uri.parse("imto://jabber/" + jid));
	    i.putExtra(BeemIntent.EXTRA_ACCOUNT, mAccountName);
	    startActivity(i);
	}
    }

    /**
     * Event simple click on middle group name.
     */
    private class OnItemClickGroupName implements OnItemClickListener {

	/**
	 * Constructor.
	 */
	public OnItemClickGroupName() {
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onItemClick(AdapterView<?> arg0, View v, int i, long l) {
	    Cursor cursor = null;
	    ListView listView = (ListView) findViewById(R.id.contactlist);
	    if (i == BANNER_POS_ALL_CONTACT) {
		cursor = getContacts();
		listView.setOnItemClickListener(mOnContactClick);
		mAdapterContactList = new BeemContactList(getBaseContext(), R.layout.contactlistcontact, cursor,
		    new String[] { ContactsContract.RawContacts._ID, ContactsContract.RawContacts.CONTACT_ID,
			ContactsContract.RawContacts.SOURCE_ID }, new int[] { R.id.avatar, R.id.contactlistpseudo,
			R.id.contactlistmsgperso });
	    } else if (i == BANNER_POS_NO_GROUP) {
		String noGroupID = getNoGroupID();
		if (noGroupID != null)
		    cursor = getContactsByGroupID(noGroupID);
		listView.setOnItemClickListener(mOnContactClick);
		mAdapterContactList = new BeemContactList(getBaseContext(), R.layout.contactlistcontact, cursor,
		    new String[] { ContactsContract.RawContacts._ID, ContactsContract.RawContacts.CONTACT_ID,
			ContactsContract.RawContacts.SOURCE_ID }, new int[] { R.id.avatar, R.id.contactlistpseudo,
			R.id.contactlistmsgperso });
	    } else if (i == BANNER_POS_MUC) {
		cursor = getMucs();
		listView.setOnItemClickListener(mOnMucClick);
		mAdapterContactList = new BeemMucList(getBaseContext(), R.layout.contactlistcontact, cursor,
		    new String[] { MUCs._ID, MUCs.NAME, MUCs.JID }, new int[] { R.id.avatar, R.id.contactlistpseudo,
			R.id.contactlistmsgperso });
	    } else {
		String groupid = v.getTag().toString();
		cursor = getContactsByGroupID(groupid);
		listView.setOnItemClickListener(mOnContactClick);
		mAdapterContactList = new BeemContactList(getBaseContext(), R.layout.contactlistcontact, cursor,
		    new String[] { ContactsContract.RawContacts._ID, ContactsContract.RawContacts.CONTACT_ID,
			ContactsContract.RawContacts.SOURCE_ID }, new int[] { R.id.avatar, R.id.contactlistpseudo,
			R.id.contactlistmsgperso });
	    }
	    if (cursor != null) {
		registerForContextMenu(listView);
		listView.setAdapter(mAdapterContactList);
		mAdapterContactList.changeCursor(cursor);
		mAdapterContactList.notifyDataSetChanged();
	    }
	}
    }

    /**
     * Class to observe modification in groups.
     */
    private class BeemGroupObserver extends ContentObserver {

	/**
	 * Constructor.
	 * @param handler
	 */
	public BeemGroupObserver(Handler handler) {
	    super(handler);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean deliverSelfNotifications() {
	    return true;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onChange(boolean selfChange) {
	    super.onChange(selfChange);
	    View v = findViewById(R.id.contactlist_groupstub);
	    if (v != null) {
		if (mAdapterBanner.getCount() > 0) {
		    v.setVisibility(View.VISIBLE);
		} else {
		    v.setVisibility(View.GONE);
		}
	    }
	}
    }

    /**
     * Retrieve contacts of the current account.
     * @return a cursor of contacts
     */
    public Cursor getContacts() {
	return getContentResolver().query(
	    ContactsContract.RawContacts.CONTENT_URI,
	    new String[] { ContactsContract.RawContacts._ID, ContactsContract.RawContacts.CONTACT_ID,
		ContactsContract.RawContacts.SOURCE_ID },
	    ContactsContract.RawContacts.ACCOUNT_NAME + "=? AND " + ContactsContract.RawContacts.ACCOUNT_TYPE
		+ "=? AND " + ContactsContract.RawContacts.DELETED + "= 0",
	    new String[] { mAccountName, BeemApplication.BEEM_PACKAGE },
	    ContactsContract.RawContacts.CONTACT_ID + " ASC ");
    }

    /**
     * Retrieve contacts by groupid.
     * @param groupid the groupid to select
     * @return a cursor of contacts
     */
    public Cursor getContactsByGroupID(String groupid) {
	return getContentResolver().query(
	    ContactsContract.Data.CONTENT_URI,
	    new String[] { ContactsContract.Data._ID, ContactsContract.RawContacts.CONTACT_ID,
		ContactsContract.RawContacts.SOURCE_ID },
	    ContactsContract.CommonDataKinds.GroupMembership.GROUP_ROW_ID + "=?", new String[] { groupid }, null);
    }

    /**
     * Get the NoGroup ID of the current account.
     * @return the ID of the NoGroup group
     */
    public String getNoGroupID() {
	final Cursor cursorGroup = getContentResolver().query(
	    ContactsContract.Groups.CONTENT_URI,
	    new String[] { ContactsContract.Groups._ID, ContactsContract.Groups.TITLE },
	    ContactsContract.Groups.ACCOUNT_NAME + "=? AND " + ContactsContract.Groups.ACCOUNT_TYPE + "=? AND "
		+ ContactsContract.Groups.SYSTEM_ID + "=?",
	    new String[] { mAccountName, BeemApplication.BEEM_PACKAGE, BeemApplication.NO_GROUP }, null);
	if (cursorGroup.moveToFirst()) {
	    return cursorGroup.getString(cursorGroup.getColumnIndex(ContactsContract.Groups._ID));
	}
	return null;
    }

    public Cursor getMucs() {
	return getContentResolver().query(MUCs.CONTENT_URI, new String[] { MUCs._ID, MUCs.NAME, MUCs.JID },
	    MUCs.ACCOUNT_NAME + "=? AND " + MUCs.ACCOUNT_TYPE + "=? ",
	    new String[] { mAccountName, BeemApplication.BEEM_PACKAGE }, null);
    }
}
