/*
    BEEM is a videoconference application on the Android Platform.

    Copyright (C) 2009 by Frederic-Charles Barthelery,
                          Jean-Manuel Da Silva,
                          Nikita Kozlov,
                          Philippe Lago,
                          Jean Baptiste Vergely,
                          Vincent Veronis.

    This file is part of BEEM.

    BEEM is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    BEEM is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with BEEM.  If not, see <http://www.gnu.org/licenses/>.

    Please send bug reports with examples or suggestions to
    contact@beem-project.com or http://dev.beem-project.com/

    Epitech, hereby disclaims all copyright interest in the program "Beem"
    written by Frederic-Charles Barthelery,
               Jean-Manuel Da Silva,
               Nikita Kozlov,
               Philippe Lago,
               Jean Baptiste Vergely,
               Vincent Veronis.

    Nicolas Sadirac, November 26, 2009
    President of Epitech.

    Flavien Astraud, November 26, 2009
    Head of the EIP Laboratory.

 */
package com.beem.project.beem;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import org.jivesoftware.smack.ConnectionConfiguration;
import org.jivesoftware.smack.Roster;
import org.jivesoftware.smack.RosterEntry;
import org.jivesoftware.smack.RosterGroup;
import org.jivesoftware.smack.XMPPConnection;
import org.jivesoftware.smack.Roster.SubscriptionMode;
import org.jivesoftware.smack.packet.Presence;
import org.jivesoftware.smack.provider.ProviderManager;
import org.jivesoftware.smack.util.StringUtils;
import org.jivesoftware.smackx.packet.ChatStateExtension;
import org.jivesoftware.smackx.provider.DelayInfoProvider;
import org.jivesoftware.smackx.provider.DiscoverInfoProvider;
import org.jivesoftware.smackx.provider.DiscoverItemsProvider;
import org.jivesoftware.smackx.pubsub.provider.EventProvider;
import org.jivesoftware.smackx.pubsub.provider.ItemProvider;
import org.jivesoftware.smackx.pubsub.provider.ItemsProvider;
import org.jivesoftware.smackx.pubsub.provider.PubSubProvider;

import android.accounts.Account;
import android.accounts.AccountManager;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.Service;
import android.content.BroadcastReceiver;
import android.content.ContentProviderOperation;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.OperationApplicationException;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.database.Cursor;
import android.net.ConnectivityManager;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.IBinder;
import android.os.Looper;
import android.os.Message;
import android.os.RemoteException;
import android.provider.ContactsContract;
import android.util.Log;

import com.beem.project.beem.service.BeemAvatarCache;
import com.beem.project.beem.service.RosterAdapter;
import com.beem.project.beem.service.XmppConnectionAdapter;
import com.beem.project.beem.service.XmppFacade;
import com.beem.project.beem.service.aidl.IRoster;
import com.beem.project.beem.service.aidl.IXmppFacade;
import com.beem.project.beem.smack.avatar.AvatarMetadataProvider;
import com.beem.project.beem.smack.avatar.AvatarProvider;
import com.beem.project.beem.smack.caps.CapsProvider;
import com.beem.project.beem.utils.BeemBroadcastReceiver;
import com.beem.project.beem.utils.BeemConnectivity;
import com.beem.project.beem.utils.Status;

/**
 * This class is for the Beem service. It must contains every global informations needed to maintain the background
 * service. The connection to the xmpp server will be made asynchronously when the service will start.
 * @author darisk
 */
public class BeemService extends Service {

    /** The id to use for status notification. */
    public static final int NOTIFICATION_STATUS_ID = 100;

    private static final String TAG = "BeemService";
    private static final int MESSAGE_CONNECT = 0x1;
    private static final int MESSAGE_SEND_MSG = 0x2;
    private static final int MESSAGE_SYNC = 0x3;
    private static final int NB_DB_OPERATION = 50;

    private NotificationManager mNotificationManager;
    private Map<String, XmppConnectionAdapter> mConnection = new HashMap<String, XmppConnectionAdapter>();
    private Map<String, BeemConnection> mBeemConnection = new HashMap<String, BeemConnection>();
    private Map<String, IXmppFacade.Stub> mBind = new HashMap<String, IXmppFacade.Stub>();

    private boolean mOnOffReceiverIsRegistered;
    private Handler mHandler;
    private Looper mServiceLooper;
    private BeemBroadcastReceiver mReceiver = new BeemBroadcastReceiver();
    private BeemServicePreferenceListener mPreferenceListener = new BeemServicePreferenceListener();
    private BeemServiceBroadcastReceiver mOnOffReceiver = new BeemServiceBroadcastReceiver();

    /**
     * Constructor.
     */
    public BeemService() {
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
	if (intent != null) {
	    handleIntent(intent);
	}
	return Service.START_STICKY;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public IBinder onBind(Intent intent) {
	Log.d(TAG, "ONBIND()");
	String accountName = intent.getStringExtra("account_name");
	Log.e(TAG, accountName);
	//	try {
	//	    for (XmppConnectionAdapter connection : mConnection.values())
	//		connection.connect();
	//	} catch (RemoteException e) {
	//	    Log.w(TAG, "Error while connecting", e);
	//	}
	return (IBinder) mBind.get(accountName);
    }

    @Override
    public boolean onUnbind(Intent intent) {
	Log.d(TAG, "ONUNBIND()");
	boolean isConnected = true;
	for (XmppConnectionAdapter connection : mConnection.values()) {
	    if (!connection.getAdaptee().isConnected())
		isConnected = false;
	}
	if (!isConnected) {
	    this.stopSelf();
	}
	return true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onCreate() {
	super.onCreate();

	HandlerThread thread = new HandlerThread("BeemServiceThread");
	thread.start();
	mServiceLooper = thread.getLooper();
	mHandler = new BeemServiceHandler(mServiceLooper);

	//	AccountManager am = AccountManager.get(BeemService.this);
	//	Account allAccount[] = am.getAccountsByType("com.beem.project.com");
	//	for (Account account : allAccount) {
	//	    BeemConnection beemco = new BeemConnection(BeemService.this
	//		.getSharedPreferences(account.name, MODE_PRIVATE), mPreferenceListener);
	//	    if (beemco.getSettings().getBoolean("settings_away_chk", false)) {
	//		mOnOffReceiverIsRegistered = true;
	//		registerReceiver(mOnOffReceiver, new IntentFilter(Intent.ACTION_SCREEN_OFF));
	//		registerReceiver(mOnOffReceiver, new IntentFilter(Intent.ACTION_SCREEN_ON));
	//	    }
	//	    mBeemConnection.put(account.name, beemco);
	//	    XmppConnectionAdapter beemcoAdapter = new XmppConnectionAdapter(beemco.getConnectionConfiguration(), beemco
	//		.getJid(), beemco.getPassword(), this);
	//	    mConnection.put(account.name, beemcoAdapter);
	//	    mBind.put(account.name, new XmppFacade(beemcoAdapter));
	//	    Log.e(TAG, "Account configuration : " + account.toString() + " DONE");
	//	}
	registerReceiver(mReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));

	configure(ProviderManager.getInstance());
	mNotificationManager = (NotificationManager) getSystemService(NOTIFICATION_SERVICE);
	Roster.setDefaultSubscriptionMode(SubscriptionMode.manual);
	Log.d(TAG, "ONCREATE");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onDestroy() {
	super.onDestroy();
	mNotificationManager.cancelAll();
	unregisterReceiver(mReceiver);

	if (mOnOffReceiverIsRegistered)
	    unregisterReceiver(mOnOffReceiver);
	for (XmppConnectionAdapter connection : mConnection.values()) {
	    if (connection.isAuthentificated() && BeemConnectivity.isConnected(this))
		connection.disconnect();
	}
	Log.d(TAG, "ONDESTROY");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onStart(Intent intent, int startId) {
	super.onStart(intent, startId);
	Log.d(TAG, "onStart");
	//	try {
	//	    for (XmppConnectionAdapter connection : mConnection.values())
	//		connection.connectAsync();
	//	} catch (RemoteException e) {
	//	    e.printStackTrace();
	//	}
    }

    public XmppConnectionAdapter getConnection(String accountName) {
	return mConnection.get(accountName);
    }

    /**
     * Show a notification using the preference of the user.
     * @param id the id of the notification.
     * @param notif the notification to show
     */
    public void sendNotification(int id, Notification notif) {
	//	if (mSettings.getBoolean(BeemApplication.NOTIFICATION_VIBRATE_KEY, true))
	//	    notif.defaults |= Notification.DEFAULT_VIBRATE;
	//	notif.defaults |= Notification.DEFAULT_LIGHTS;
	//	String ringtoneStr = mSettings.getString(BeemApplication.NOTIFICATION_SOUND_KEY, "");
	//	notif.sound = Uri.parse(ringtoneStr);
	//	mNotificationManager.notify(id, notif);
    }

    /**
     * Delete a notification.
     * @param id the id of the notification
     */
    public void deleteNotification(int id) {
	mNotificationManager.cancel(id);
    }

    /**
     * Reset the status to online after a disconnect.
     */
    public void resetStatus() {
	for (BeemConnection beemco : mBeemConnection.values()) {
	    Editor edit = beemco.getSettings().edit();
	    edit.putInt(BeemApplication.STATUS_KEY, 1);
	    edit.commit();
	}

    }

    /**
     * Initialize Jingle from an XmppConnectionAdapter.
     * @param adaptee XmppConnection used for jingle.
     */
    public void initJingle(XMPPConnection adaptee) {
    }

    /**
     * Return a bind to an XmppFacade instance.
     * @return IXmppFacade a bind to an XmppFacade instance
     */
    public IXmppFacade getBind() {
	//TODO: Prendre le bon mbind dans le tableau
	Log.e(TAG, "GETBIND");
	return mBind.get(0);
    }

    /**
     * Get the preference of the service.
     * @return the preference
     */
    public SharedPreferences getServicePreference(String accountName) {
	return mBeemConnection.get(accountName).getSettings();
    }

    /**
     * Get the notification manager system service.
     * @return the notification manager service.
     */
    public NotificationManager getNotificationManager() {
	return mNotificationManager;
    }

    /**
     * A sort of patch from this thread: http://www.igniterealtime.org/community/thread/31118. Avoid ClassCastException
     * by bypassing the classloading shit of Smack.
     * @param pm The ProviderManager.
     */
    private void configure(ProviderManager pm) {
	Log.d(TAG, "configure");
	// Service Discovery # Items
	pm.addIQProvider("query", "http://jabber.org/protocol/disco#items", new DiscoverItemsProvider());
	// Service Discovery # Info
	pm.addIQProvider("query", "http://jabber.org/protocol/disco#info", new DiscoverInfoProvider());

	// Privacy
	//pm.addIQProvider("query", "jabber:iq:privacy", new PrivacyProvider());
	// Delayed Delivery only the new version
	pm.addExtensionProvider("delay", "urn:xmpp:delay", new DelayInfoProvider());

	// Service Discovery # Items
	pm.addIQProvider("query", "http://jabber.org/protocol/disco#items", new DiscoverItemsProvider());
	// Service Discovery # Info
	pm.addIQProvider("query", "http://jabber.org/protocol/disco#info", new DiscoverInfoProvider());

	// Chat State
	ChatStateExtension.Provider chatState = new ChatStateExtension.Provider();
	pm.addExtensionProvider("active", "http://jabber.org/protocol/chatstates", chatState);
	pm.addExtensionProvider("composing", "http://jabber.org/protocol/chatstates", chatState);
	pm.addExtensionProvider("paused", "http://jabber.org/protocol/chatstates", chatState);
	pm.addExtensionProvider("inactive", "http://jabber.org/protocol/chatstates", chatState);
	pm.addExtensionProvider("gone", "http://jabber.org/protocol/chatstates", chatState);
	// capabilities
	pm.addExtensionProvider("c", "http://jabber.org/protocol/caps", new CapsProvider());
	//Pubsub
	pm.addIQProvider("pubsub", "http://jabber.org/protocol/pubsub", new PubSubProvider());
	pm.addExtensionProvider("items", "http://jabber.org/protocol/pubsub", new ItemsProvider());
	pm.addExtensionProvider("items", "http://jabber.org/protocol/pubsub", new ItemsProvider());
	pm.addExtensionProvider("item", "http://jabber.org/protocol/pubsub", new ItemProvider());

	pm.addExtensionProvider("items", "http://jabber.org/protocol/pubsub#event", new ItemsProvider());
	pm.addExtensionProvider("item", "http://jabber.org/protocol/pubsub#event", new ItemProvider());
	pm.addExtensionProvider("event", "http://jabber.org/protocol/pubsub#event", new EventProvider());
	//TODO rajouter les manquants pour du full pubsub

	//PEP avatar
	pm.addExtensionProvider("metadata", "urn:xmpp:avatar:metadata", new AvatarMetadataProvider());
	pm.addExtensionProvider("data", "urn:xmpp:avatar:data", new AvatarProvider());

	//         PEPProvider pep  = new PEPProvider();
	//         AvatarMetadataProvider avaMeta  = new AvatarMetadataProvider();
	//         pep.registerPEPParserExtension("urn:xmpp:avatar:metadata", avaMeta);
	//         pm.addExtensionProvider("event", "http://jabber.org/protocol/pubsub#event", pep);

	/*
	 * // Private Data Storage pm.addIQProvider("query", "jabber:iq:private", new
	 * PrivateDataManager.PrivateDataIQProvider()); // Time try { pm.addIQProvider("query", "jabber:iq:time",
	 * Class.forName("org.jivesoftware.smackx.packet.Time")); } catch (ClassNotFoundException e) {
	 * Log.w("TestClient", "Can't load class for org.jivesoftware.smackx.packet.Time"); } // Roster Exchange
	 * pm.addExtensionProvider("x", "jabber:x:roster", new RosterExchangeProvider()); // Message Events
	 * pm.addExtensionProvider("x", "jabber:x:event", new MessageEventProvider()); // XHTML
	 * pm.addExtensionProvider("html", "http://jabber.org/protocol/xhtml-im", new XHTMLExtensionProvider()); //
	 * Group Chat Invitations pm.addExtensionProvider("x", "jabber:x:conference", new
	 * GroupChatInvitation.Provider()); // Data Forms pm.addExtensionProvider("x", "jabber:x:data", new
	 * DataFormProvider()); // MUC User pm.addExtensionProvider("x", "http://jabber.org/protocol/muc#user", new
	 * MUCUserProvider()); // MUC Admin pm.addIQProvider("query", "http://jabber.org/protocol/muc#admin", new
	 * MUCAdminProvider()); // MUC Owner pm.addIQProvider("query", "http://jabber.org/protocol/muc#owner", new
	 * MUCOwnerProvider()); // Version try { pm.addIQProvider("query", "jabber:iq:version",
	 * Class.forName("org.jivesoftware.smackx.packet.Version")); } catch (ClassNotFoundException e) { // Not sure
	 * what's happening here. Log.w("TestClient", "Can't load class for org.jivesoftware.smackx.packet.Version"); }
	 * // VCard pm.addIQProvider("vCard", "vcard-temp", new VCardProvider()); // Offline Message Requests
	 * pm.addIQProvider("offline", "http://jabber.org/protocol/offline", new OfflineMessageRequest.Provider()); //
	 * Offline Message Indicator pm.addExtensionProvider("offline", "http://jabber.org/protocol/offline", new
	 * OfflineMessageInfo.Provider()); // Last Activity pm.addIQProvider("query", "jabber:iq:last", new
	 * LastActivity.Provider()); // User Search pm.addIQProvider("query", "jabber:iq:search", new
	 * UserSearch.Provider()); // SharedGroupsInfo pm.addIQProvider("sharedgroup",
	 * "http://www.jivesoftware.org/protocol/sharedgroup", new SharedGroupsInfo.Provider()); // JEP-33: Extended
	 * Stanza Addressing pm.addExtensionProvider("addresses", "http://jabber.org/protocol/address", new
	 * MultipleAddressesProvider()); // FileTransfer pm.addIQProvider("si", "http://jabber.org/protocol/si", new
	 * StreamInitiationProvider()); pm.addIQProvider("query", "http://jabber.org/protocol/bytestreams", new
	 * BytestreamsProvider()); pm.addIQProvider("open", "http://jabber.org/protocol/ibb", new IBBProviders.Open());
	 * pm.addIQProvider("close", "http://jabber.org/protocol/ibb", new IBBProviders.Close());
	 * pm.addExtensionProvider("data", "http://jabber.org/protocol/ibb", new IBBProviders.Data());
	 * pm.addIQProvider("command", COMMAND_NAMESPACE, new AdHocCommandDataProvider());
	 * pm.addExtensionProvider("malformed-action", COMMAND_NAMESPACE, new
	 * AdHocCommandDataProvider.MalformedActionError()); pm.addExtensionProvider("bad-locale", COMMAND_NAMESPACE,
	 * new AdHocCommandDataProvider.BadLocaleError()); pm.addExtensionProvider("bad-payload", COMMAND_NAMESPACE, new
	 * AdHocCommandDataProvider.BadPayloadError()); pm.addExtensionProvider("bad-sessionid", COMMAND_NAMESPACE, new
	 * AdHocCommandDataProvider.BadSessionIDError()); pm.addExtensionProvider("session-expired", COMMAND_NAMESPACE,
	 * new AdHocCommandDataProvider.SessionExpiredError());
	 */
    }

    /**
     * Listen on preference changes.
     */
    public class BeemServicePreferenceListener implements SharedPreferences.OnSharedPreferenceChangeListener {

	/**
	 * ctor.
	 */
	public BeemServicePreferenceListener() {
	}

	@Override
	public void onSharedPreferenceChanged(SharedPreferences sharedPreferences, String key) {
	    if ("settings_away_chk".equals(key)) {
		if (sharedPreferences.getBoolean("settings_away_chk", false)) {
		    mOnOffReceiverIsRegistered = true;
		    registerReceiver(mOnOffReceiver, new IntentFilter(Intent.ACTION_SCREEN_OFF));
		    registerReceiver(mOnOffReceiver, new IntentFilter(Intent.ACTION_SCREEN_ON));
		} else {
		    mOnOffReceiverIsRegistered = false;
		    unregisterReceiver(mOnOffReceiver);
		}
	    }
	}
    }

    /**
     * Listen on some Intent broadcast, ScreenOn and ScreenOff.
     */
    private class BeemServiceBroadcastReceiver extends BroadcastReceiver {

	private String mOldStatus;
	private int mOldMode;

	/**
	 * Constructor.
	 */
	public BeemServiceBroadcastReceiver() {
	}

	@Override
	public void onReceive(final Context context, final Intent intent) {
	    String intentAction = intent.getAction();
	    if (intentAction.equals(Intent.ACTION_SCREEN_OFF)) {
		for (Map.Entry<String, XmppConnectionAdapter> item : mConnection.entrySet()) {
		    XmppConnectionAdapter connection = item.getValue();
		    mOldMode = connection.getPreviousMode();
		    mOldStatus = connection.getPreviousStatus();
		    if (connection.isAuthentificated())
			connection.changeStatus(Status.CONTACT_STATUS_AWAY, mBeemConnection.get(item.getKey())
			    .getSettings().getString("settings_away_message", "Away"));
		}
	    } else if (intentAction.equals(Intent.ACTION_SCREEN_ON)) {
		for (XmppConnectionAdapter connection : mConnection.values()) {
		    if (connection.isAuthentificated())
			connection.changeStatus(mOldMode, mOldStatus);
		}
	    }
	}
    }

    private void handleIntent(Intent intent) {
	Message msg = null;
	String action = intent.getAction();
	if (BeemIntent.ACTION_CONNECT.equals(action)) {
	    msg = mHandler.obtainMessage(MESSAGE_CONNECT, intent.getExtras());
	} else if (BeemIntent.ACTION_SEND_MESSAGE.equals(action)) {
	    msg = mHandler.obtainMessage(MESSAGE_SEND_MSG, intent.getExtras());
	} else if (BeemIntent.ACTION_SYNC.equals(action)) {
	    msg = mHandler.obtainMessage(MESSAGE_SYNC, intent.getExtras());
	} else {
	    Log.w(TAG, "Unknown intent " + intent);
	}
	if (msg != null)
	    mHandler.sendMessage(msg);
    }

    private class BeemServiceHandler extends Handler {

	public BeemServiceHandler(Looper looper) {
	    super(looper);
	}

	@Override
	public void handleMessage(Message msg) {
	    Bundle b = (Bundle) msg.obj;
	    switch (msg.what) {
		case MESSAGE_CONNECT:
		    handleConnect(b);
		    break;
		case MESSAGE_SEND_MSG:
		    String account = b.getString(BeemIntent.EXTRA_ACCOUNT);
		    XmppConnectionAdapter con = mConnection.get(account);
		    if (con != null) {
			con.handleMessage(msg);
		    }
		    break;
		case MESSAGE_SYNC:		 
		    Account a = b.getParcelable(BeemIntent.EXTRA_ACCOUNT);
		    Bundle accountName = new Bundle();
		    accountName.putString(BeemIntent.EXTRA_ACCOUNT, a.name);
		    handleConnect(accountName);
		    XmppConnectionAdapter co = mConnection.get(a.name);
		    if (co != null) {
		        manageRoster(co.getAdaptee().getRoster(), a);
		    }
		    break;
		default:
		    Log.w(TAG, "Unknown message " + msg);
	    }
	}
    }

    private void handleConnect(Bundle b) {
	Intent res = new Intent(BeemIntent.ACTION_DISCONNECTED);
	String account = null;
	if (b != null)
	    account = b.getString(BeemIntent.EXTRA_ACCOUNT);
	if (account == null) //TODO temporary
	    account = "dummy";
	if (account == null) {
	    //connect all
	} else {

	    BeemConnection beemco = new BeemConnection(BeemService.this.getSharedPreferences(account, MODE_PRIVATE),
		mPreferenceListener);
	    if (beemco.getSettings().getBoolean("settings_away_chk", false)) {
		mOnOffReceiverIsRegistered = true;
		registerReceiver(mOnOffReceiver, new IntentFilter(Intent.ACTION_SCREEN_OFF));
		registerReceiver(mOnOffReceiver, new IntentFilter(Intent.ACTION_SCREEN_ON));
	    }
	    mBeemConnection.put(account, beemco);
	    XmppConnectionAdapter beemcoAdapter = new XmppConnectionAdapter(beemco.getConnectionConfiguration(), beemco
		.getJid(), beemco.getPassword(), this);

	    try {
		Log.i(TAG, "Starting connection of " + account);
		if (beemcoAdapter.connectSync()) {
		    mConnection.put(account, beemcoAdapter);
		    mBind.put(account, new XmppFacade(beemcoAdapter));

		    res.setAction(BeemIntent.ACTION_CONNECTED);
		    Log.e(TAG, "Account configuration : " + account.toString() + " DONE");
		} else {
		    Log.w(TAG, "Unable to connect " + account);
		}
	    } catch (RemoteException e) {
		Log.e(TAG, "Unable to connect " + account, e);
	    }

	    //	    res.putExtra(BeemIntent.EXTRA_ACCOUNT, account);
	    //	    XmppConnectionAdapter con = mConnections.get(account);
	    //	    if (con != null && con.isAuthentificated())
	    //		return;
	    //	    if (con == null ) {
	    //		ConnectionConfiguration config = initConnectionConfig(account);
	    //		String login = StringUtils.parseName(account);
	    //		String password = mSettings.getString(BeemApplication.ACCOUNT_PASSWORD_KEY, "");
	    //
	    //		if (mSettings.getBoolean(BeemApplication.FULL_JID_LOGIN_KEY, false) ||
	    //			"gmail.com".equals(mService) || "googlemail.com".equals(mService))  {
	    //		    login = account;
	    //			}
	    //		con = new XmppConnectionAdapter(config, login, password, this);
	    //		try {
	    //		    Log.i(TAG, "Starting connection of " + account);
	    //		    if(con.connectSync()){
	    //			mConnections.put(account, con);
	    //			mConnection = con;
	    //			
	    //			//TODO 
	    //			mBind = new XmppFacade(con);
	    //		    } else {
	    //			Log.w(TAG, "Unable to connect " + account);
	    //		    }
	    //		} catch (RemoteException e) {
	    //		    Log.e(TAG, "Unable to connect " + account, e);
	    //	    }
	    //	}
	    sendBroadcast(res);
	}
    }

    /**
     * Method to execute content provider operation.
     * @param ops
     */
    private void executeOperation(final ArrayList<ContentProviderOperation> ops) {
	try {
	    getContentResolver().applyBatch(ContactsContract.AUTHORITY, ops);
	} catch (RemoteException e) {
	    Log.d(TAG, "Error during sync of contact", e);
	} catch (OperationApplicationException e) {
	    Log.d(TAG, "Error during sync of contact", e);
	}
	ops.clear();
    }

    /**
     * Roster sync method.
     * @param r The roster to sync
     * @param a The account related
     */
    private void manageRoster(final Roster r, final Account a) {
	Log.e("OOO", "roster" + r + "account" + a);
	ArrayList<ContentProviderOperation> ops = new ArrayList<ContentProviderOperation>();
	for (RosterGroup group : r.getGroups()) {
	    if (group != null) {
		manageGroup(ops, a, group);
	    }
	    if (ops.size() > NB_DB_OPERATION)
		executeOperation(ops);
	}
	if (ops.size() > 0)
	    executeOperation(ops);
	for (RosterEntry entry : r.getEntries()) {
	    if (entry != null) {
		long rawContactID = manageEntry(ops, a, entry);
		addUpdateStatus(ops, entry, r.getPresence(entry.getUser()), rawContactID);
	    }
	    if (ops.size() > NB_DB_OPERATION)
		executeOperation(ops);
	}
	if (ops.size() > 0)
	    executeOperation(ops);
    }

    private void manageGroup(ArrayList<ContentProviderOperation> ops, Account account, RosterGroup group) {
	Log.i(TAG, "Sync group : " + group.getName() + " " + group.getEntryCount());
	long rawGroupID = getRawGroupID(account.name, group.getName());
	if (rawGroupID == -1) {
	    ContentProviderOperation.Builder builder = ContentProviderOperation
		.newInsert(ContactsContract.Groups.CONTENT_URI);
	    builder.withValue(ContactsContract.Groups.ACCOUNT_NAME, account.name);
	    builder.withValue(ContactsContract.Groups.ACCOUNT_TYPE, account.type);
	    builder.withValue(ContactsContract.Groups.TITLE, group.getName());
	    ops.add(builder.build());
	}
    }

    /**
     * RosterEntry sync method.
     * @param ops The content provider operation
     * @param account The account related
     * @param entry The roster entry to sync
     * @return The raw contact ID
     */
    private long manageEntry(ArrayList<ContentProviderOperation> ops, Account account, RosterEntry entry) {
	long rawContactID = getRawContactID(account.name, entry.getUser());
	Log.i(TAG, "Sync Contact : " + entry.getUser() + " RawContactID : " + rawContactID);
	if (rawContactID == -1) { // Not found in database, add new
	    ContentValues values = new ContentValues();
	    values.put(ContactsContract.RawContacts.ACCOUNT_TYPE, account.type);
	    values.put(ContactsContract.RawContacts.ACCOUNT_NAME, account.name);
	    values.put(ContactsContract.RawContacts.SOURCE_ID, entry.getUser());
	    Uri rawContactUri = getContentResolver().insert(ContactsContract.RawContacts.CONTENT_URI, values);
	    rawContactID = ContentUris.parseId(rawContactUri);
	    values.clear();
	    ContentProviderOperation.Builder builder = addUpdateStructuredName(entry, rawContactID, true);
	    ops.add(builder.build());
	    for (RosterGroup group : entry.getGroups()) {
		builder = addUpdateGroup(entry, rawContactID, getRawGroupID(account.name, group.getName()), true);
		ops.add(builder.build());
	    }
	    builder = createProfile(entry, rawContactID, account);
	    ops.add(builder.build());
	} else { // Found, update	   
	    ContentProviderOperation.Builder builder = addUpdateStructuredName(entry, rawContactID, false);
	    ops.add(builder.build());
	    //TODO: ADD AVATAR
	    //builder = addUpdatePhoto(entry, rawContactID, false);
	    //ops.add(builder.build());
	}
	return rawContactID;
    }

    /**
     * Method to insert or update structured name informations.
     * @param entry The roster entry to sync
     * @param rawContactID The contact ID in the android database
     * @param isInsert Insert boolean
     * @return
     */
    private ContentProviderOperation.Builder addUpdateStructuredName(RosterEntry entry, long rawContactID,
	boolean isInsert) {
	String displayName = entry.getName() != null ? entry.getName() : entry.getUser();
	ContentProviderOperation.Builder builder;
	if (isInsert) {
	    builder = ContentProviderOperation.newInsert(ContactsContract.Data.CONTENT_URI);
	    builder.withValue(ContactsContract.Data.RAW_CONTACT_ID, rawContactID);
	    builder.withValue(ContactsContract.Data.MIMETYPE,
		ContactsContract.CommonDataKinds.StructuredName.CONTENT_ITEM_TYPE);
	    builder.withValue(ContactsContract.CommonDataKinds.StructuredName.RAW_CONTACT_ID, rawContactID);
	} else {
	    builder = ContentProviderOperation.newUpdate(ContactsContract.Data.CONTENT_URI);
	    builder.withSelection(
		ContactsContract.CommonDataKinds.StructuredName.RAW_CONTACT_ID + " =? AND "
		    + ContactsContract.Data.MIMETYPE + " = '" + ContactsContract.CommonDataKinds.Im.CONTENT_ITEM_TYPE
		    + "'", new String[] { String.valueOf(rawContactID) });
	}
	builder.withValue(ContactsContract.CommonDataKinds.StructuredName.DISPLAY_NAME, displayName);
	builder.withValue(ContactsContract.CommonDataKinds.StructuredName.GIVEN_NAME, displayName);
	builder.withValue(ContactsContract.CommonDataKinds.StructuredName.FAMILY_NAME, displayName);
	return builder;
    }

    private ContentProviderOperation.Builder addUpdatePhoto(RosterEntry entry, long rawContactID, boolean isInsert) {
	String displayName = entry.getName() != null ? entry.getName() : entry.getUser();
	ContentProviderOperation.Builder builder;
	if (isInsert) {
	    builder = ContentProviderOperation.newInsert(ContactsContract.Data.CONTENT_URI);
	    builder.withValue(ContactsContract.Data.RAW_CONTACT_ID, rawContactID);
	    builder.withValue(ContactsContract.Data.MIMETYPE, ContactsContract.CommonDataKinds.Photo.CONTENT_ITEM_TYPE);
	    builder.withValue(ContactsContract.CommonDataKinds.Photo.RAW_CONTACT_ID, rawContactID);
	} else {
	    builder = ContentProviderOperation.newUpdate(ContactsContract.Data.CONTENT_URI);
	    builder.withSelection(ContactsContract.CommonDataKinds.Photo.RAW_CONTACT_ID + " =? AND "
		+ ContactsContract.Data.MIMETYPE + " = '" + ContactsContract.CommonDataKinds.Photo.CONTENT_ITEM_TYPE
		+ "'", new String[] { String.valueOf(rawContactID) });
	}
	BeemAvatarCache bac = new BeemAvatarCache(getBaseContext());
	try {
	    builder.withValue(ContactsContract.CommonDataKinds.Photo.PHOTO, bac.get(entry.getUser()));
	} catch (IOException e) {
	    Log.e(TAG, "Error cache beem avatar", e);
	}
	return builder;
    }

    /**
     * Method to insert or update group name informations.
     * @param entry The roster entry to sync
     * @param rawContactID The contact ID in the android database
     * @param rawGroupID The group ID in the android database
     * @param isInsert Insert boolean
     * @return
     */
    private ContentProviderOperation.Builder addUpdateGroup(RosterEntry entry, long rawContactID, long rawGroupID,
	boolean isInsert) {
	String displayName = entry.getName() != null ? entry.getName() : entry.getUser();
	Log.e(TAG + "UPDATE GROUP", "Contact : " + displayName + " GroupID :" + rawGroupID);
	ContentProviderOperation.Builder builder = null;
	if (isInsert) {
	    builder = ContentProviderOperation.newInsert(ContactsContract.Data.CONTENT_URI);
	    builder.withValue(ContactsContract.Data.RAW_CONTACT_ID, rawContactID);
	    builder.withValue(ContactsContract.Data.MIMETYPE,
		ContactsContract.CommonDataKinds.GroupMembership.CONTENT_ITEM_TYPE);
	    builder.withValue(ContactsContract.CommonDataKinds.GroupMembership.GROUP_ROW_ID, rawGroupID);
	}
	//TODO: delete - contact doesnt appear anymore in this group 
	return builder;
    }

    /**
     * Method to insert or update IM informations.
     * @param entry The roster entry to sync
     * @param rawContactID The contact ID in the android database
     * @param isInsert Insert boolean
     * @return
     */
    private ContentProviderOperation.Builder createProfile(RosterEntry entry, long rawContactID, Account account) {
	String displayName = entry.getName() != null ? entry.getName() : entry.getUser();
	ContentProviderOperation.Builder builder;
	builder = ContentProviderOperation.newInsert(ContactsContract.Data.CONTENT_URI);
	builder.withValue(ContactsContract.Data.RAW_CONTACT_ID, rawContactID);
	builder.withValue(ContactsContract.Data.MIMETYPE, ContactsContract.CommonDataKinds.Im.CONTENT_ITEM_TYPE);
	builder.withValue(ContactsContract.CommonDataKinds.Im.RAW_CONTACT_ID, rawContactID);
	builder.withValue(ContactsContract.CommonDataKinds.Im.DATA1, displayName);
	builder.withValue(ContactsContract.CommonDataKinds.Im.PROTOCOL,
	    ContactsContract.CommonDataKinds.Im.PROTOCOL_JABBER);
	return builder;
    }

    /**
     * Method to insert or update IM informations.
     * @param entry The roster entry to sync
     * @param rawContactID The contact ID in the android database
     * @param isInsert Insert boolean
     * @return
     */
    private void addUpdateStatus(ArrayList<ContentProviderOperation> ops, RosterEntry entry, Presence p,
	long rawContactID) {
	String displayName = entry.getName() != null ? entry.getName() : entry.getUser();
	Log.i(TAG + "UPDATESTATUS", "Contact : " + displayName + " Presence status : " + p.getStatus()
	    + " Presence status state : " + Status.getStatusFromPresence(p));
	ContentProviderOperation.Builder builder;
	builder = ContentProviderOperation.newInsert(ContactsContract.StatusUpdates.CONTENT_URI);
	builder.withValue(ContactsContract.StatusUpdates.PROTOCOL, ContactsContract.CommonDataKinds.Im.PROTOCOL_JABBER);
	builder.withValue(ContactsContract.StatusUpdates.IM_HANDLE, displayName);
	//TODO: Get account name
	builder.withValue(ContactsContract.StatusUpdates.IM_ACCOUNT, "beem@elyzion.net");
	builder.withValue(ContactsContract.StatusUpdates.STATUS, p.getStatus());
	builder.withValue(ContactsContract.StatusUpdates.STATUS_RES_PACKAGE, "com.beem.project.beem");
	builder.withValue(ContactsContract.StatusUpdates.STATUS_LABEL, R.string.app_name);
	//TODO: Get status icon
	builder.withValue(ContactsContract.StatusUpdates.STATUS_ICON, R.drawable.beem_status_icon);
	//TODO: Pb presence ... 2 appear on 3 raw .... random appear
	builder.withValue(ContactsContract.StatusUpdates.PRESENCE, Status.getStatusFromPresence(p));
	ops.add(builder.build());
    }

    /**
     * Get contact ID from android database.
     * @param account The account related
     * @param jid The jid related
     * @return ID in the database of the jid
     */
    private long getRawContactID(String account, String jid) {
	long authorId = -1;
	final Cursor c = getContentResolver().query(ContactsContract.RawContacts.CONTENT_URI,
	    new String[] { ContactsContract.RawContacts._ID, ContactsContract.RawContacts.SOURCE_ID },
	    ContactsContract.RawContacts.ACCOUNT_NAME + "=? AND " + ContactsContract.RawContacts.SOURCE_ID + "=?",
	    new String[] { account, jid }, null);
	try {
	    if (c.moveToFirst())
		authorId = c.getInt(c.getColumnIndex(ContactsContract.RawContacts._ID));
	} finally {
	    if (c != null)
		c.close();
	}
	return authorId;
    }

    /**
     * Get group ID from android database.
     * @param account The account related
     * @param group The group related
     * @return ID in the database of the jid
     */
    private long getRawGroupID(String account, String group) {
	long authorId = -1;
	final Cursor c = getContentResolver().query(ContactsContract.Groups.CONTENT_URI,
	    new String[] { ContactsContract.Groups._ID },
	    ContactsContract.Groups.ACCOUNT_NAME + "=? AND " + ContactsContract.Groups.TITLE + "=?",
	    new String[] { account, group }, null);
	try {
	    if (c.moveToFirst())
		authorId = c.getInt(c.getColumnIndex(ContactsContract.Groups._ID));
	} finally {
	    if (c != null)
		c.close();
	}
	return authorId;
    }
}
