/*
 * AboutOss is an utility library to retrieve and display
 * opensource licenses in Android applications.
 *
 * Copyright (C) 2023-2025 by Frederic-Charles Barthelery.
 *
 * This file is part of AboutOss.
 *
 * AboutOss is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * AboutOss is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with AboutOss.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.geekorum.aboutoss.sampleapp

import androidx.activity.viewModels
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.viewmodel.CreationExtras
import androidx.lifecycle.viewmodel.initializer
import androidx.lifecycle.viewmodel.viewModelFactory
import com.geekorum.aboutoss.core.gms.GmsLicenseInfoRepository
import com.geekorum.aboutoss.ui.common.AndroidBrowserLauncher
import com.geekorum.aboutoss.ui.common.OpenSourceLicensesViewModel
import com.geekorum.aboutoss.ui.material3.OpenSourceLicensesActivity
import com.geekorum.geekdroid.network.BrowserLauncher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.IO
import com.geekorum.aboutoss.ui.material.OpenSourceLicensesActivity as Material2OpenSourceLicensesActivity

/**
 * Custom activity needed to load resources from another set of files than default generated by
 * OSS Licenses Gradle Plugin.
 *
 * This is necessary because OSS License gradle plugin generates stub resources on debug builds.
 */
class PrebuiltLicencesMaterial2Activity : Material2OpenSourceLicensesActivity() {
    override val viewModel: OpenSourceLicensesViewModel by viewModels(
        factoryProducer = {
            viewModelFactory {
                initializer {
                    createPrebuildOpenSourceLicensesViewModel()
                }
            }
        }
    )
}


/**
 * Custom activity needed to load resources from another set of files than default generated by
 * OSS Licenses Gradle Plugin.
 *
 * This is necessary because OSS License gradle plugin generates stub resources on debug builds.
 */
class PrebuiltLicencesMaterial3Activity : OpenSourceLicensesActivity() {
    override val viewModel: OpenSourceLicensesViewModel by viewModels(
        factoryProducer = {
            viewModelFactory {
                initializer {
                    createPrebuildOpenSourceLicensesViewModel()
                }
            }
        }
    )
}

actual fun CreationExtras.createPrebuildOpenSourceLicensesViewModel(): OpenSourceLicensesViewModel {
    val application =
        this[ViewModelProvider.AndroidViewModelFactory.APPLICATION_KEY]!!
    val licenseInfoRepository = GmsLicenseInfoRepository(
        appContext = application,
        mainCoroutineDispatcher = Dispatchers.Main,
        ioCoroutineDispatcher = Dispatchers.IO,
        thirdPartyLicensesResourceName = "prebuilt_third_party_licenses",
        thirdPartyLicenseMetadataResourceName = "prebuilt_third_party_license_metadata"
    )
    val browserLauncher = BrowserLauncher(application, application.packageManager)
    return OpenSourceLicensesViewModel(
        licenseInfoRepository,
        AndroidBrowserLauncher(application, browserLauncher)
    )
}
